
#include "sram.h"
#include "p33Fxxxx.h"

static unsigned short current_mem_addr, pending_mem_addr;
static volatile unsigned char mem_addr_pending;

void __attribute__((__interrupt__,no_auto_psv)) _SPI1Interrupt(void) {
  IFS0bits.SPI1IF = 0;

  mem_addr_pending = 2;

  SPI1STATbits.SPIEN = 0;
}

static void send_spi_word(unsigned short word) {
  // send 16 bits of data
  SPI1STATbits.SPIEN = 1;
  SPI1BUF = word;
  // bring RCK/CS low
  LATAbits.LATA1 = 0;
}

void sram_prepare_addr(unsigned short addr) {
  mem_addr_pending = 1;
  pending_mem_addr = addr;
  send_spi_word(addr);
}

static void finish_pending_mem_change() {
  LATAbits.LATA1 = 1;
  current_mem_addr = pending_mem_addr;
  mem_addr_pending = 0;
}

void sram_change_addr(unsigned short addr) {
  if( mem_addr_pending == 1 && pending_mem_addr == addr ) {
    while( SPI1STATbits.SPIEN )
      ;
  }
  if( mem_addr_pending == 2 )
    finish_pending_mem_change();
  if( current_mem_addr != addr ) {
    sram_prepare_addr(addr);
    while( SPI1STATbits.SPIEN )
      ;
    finish_pending_mem_change();
  }
}

void sram_init() {
  // RA1 = RCK/CS (digital output, default high)
  TRISAbits.TRISA1 = 0;
  LATAbits.LATA1 = 1;
  AD1PCFGLbits.PCFG1 = 1;

  // RB15/RP15 = SDO (digital output, default high)
  // connect to SPI1 Data Output, 0x07
  AD1PCFGLbits.PCFG9 = 1;
  TRISBbits.TRISB15 = 0;
  LATBbits.LATB15 = 0;
  RPOR7bits.RP15R = 0x07;

  // RB3/RP3 = SCK (digital output, default high)
  // connect to SPI1 Clock Output, 0x08
  AD1PCFGLbits.PCFG5 = 1;
  TRISBbits.TRISB3 = 0;
  LATBbits.LATB3 = 0;
  RPOR1bits.RP3R = 0x08;

  // set up SPI1
  SPI1CON1bits.PPRE = 0x2;
  SPI1CON1bits.SPRE = 0x7;
  SPI1CON1bits.MSTEN = 1;
  SPI1CON1bits.CKP = 1;
  SPI1CON1bits.MODE16 = 1;
  IFS0bits.SPI1IF = 0;
  IEC0bits.SPI1IE = 1;

  // RA3 = memory OE-bar (digital output, default high)
  TRISAbits.TRISA3 = 0;
  LATAbits.LATA3 = 1;

  // RB4 = memory data bus (digital input)
  TRISBbits.TRISB4 = 1;

  // RA4 = memory WR-bar (digital output, default high)
  TRISAbits.TRISA4 = 0;
  LATAbits.LATA4 = 1;

  // RB5-11 = memory data bus (digital input)
  TRISBbits.TRISB5 = 1;
  TRISBbits.TRISB6 = 1;
  TRISBbits.TRISB7 = 1;
  TRISBbits.TRISB8 = 1;
  TRISBbits.TRISB9 = 1;
  TRISBbits.TRISB10 = 1;
  TRISBbits.TRISB11 = 1;

  // RB12-14 = memory address bus (digital output, default low)
  AD1PCFGLbits.PCFG12 = 1;
  TRISBbits.TRISB12 = 0;
  LATBbits.LATB12 = 0;
  AD1PCFGLbits.PCFG11 = 1;
  TRISBbits.TRISB13 = 0;
  LATBbits.LATB13 = 0;
  AD1PCFGLbits.PCFG10 = 1;
  TRISBbits.TRISB14 = 0;
  LATBbits.LATB14 = 0;

  current_mem_addr = 0xFFFF;
  sram_change_addr(0);
}

void sram_select_byte(unsigned char which) {
  LATB = (LATB&~((1<<12)|(1<<13)|(1<<14))) | ((unsigned short)which<<12);
}

void sram_write_byte(unsigned char addr, unsigned char data) {
  // data bus is RB4-RB11
  LATB = (LATB&~0x7FF0)|((unsigned short)data<<4)|((unsigned short)addr<<12);
  TRISB &= ~0xFF0;
  LATAbits.LATA4 = 0;
  asm("nop");
  asm("nop");
#ifdef EXTRA_MEM_DELAYS
  asm("nop");// Not strictly necessary but allow the device
  asm("nop");// to operate normally with the ICSP attached.
  asm("nop");// (although the memory test may still fail).
  asm("nop");//
#else
  asm("nop");// Extra cycle delay just to be safe
#endif
  TRISB |=  0xFF0;
  LATAbits.LATA4 = 1;
}

unsigned char sram_read_byte(unsigned char addr) {
  unsigned char ret;
  sram_select_byte(addr);
  LATAbits.LATA3 = 0;
  asm("nop");
  asm("nop");
  asm("nop");
  asm("nop");
#ifdef EXTRA_MEM_DELAYS
  asm("nop");// Not strictly necessary but allow the device
  asm("nop");// to operate normally with the ICSP attached.
  asm("nop");// (although the memory test may still fail).
  asm("nop");//
#else
  asm("nop");// Extra cycle delay just to be safe
#endif
  ret = PORTB>>4;
  LATAbits.LATA3 = 1;
  return ret;
}

void sram_read_data(unsigned char* dest, unsigned short page, unsigned char num_bytes) {
  unsigned char offset = 0;
  sram_change_addr(page);
  sram_prepare_addr(++page);
  while( num_bytes >= 8 ) {
    sram_change_addr(page);
    sram_prepare_addr(++page);
    dest[0] = sram_read_byte(0);
    dest[1] = sram_read_byte(1);
    dest[2] = sram_read_byte(2);
    dest[3] = sram_read_byte(3);
    dest[4] = sram_read_byte(4);
    dest[5] = sram_read_byte(5);
    dest[6] = sram_read_byte(6);
    dest[7] = sram_read_byte(7);
    dest += 8;
    num_bytes -= 8;
  }
  if( num_bytes ) {
    sram_change_addr(page);
    while( num_bytes ) {
      dest[0] = sram_read_byte(offset);
      ++offset;
      ++dest;
      --num_bytes;
    }
  }
}

void sram_write_data(unsigned short page, const unsigned char* src, unsigned char num_bytes) {
  unsigned char offset = 0;
  sram_change_addr(page);
  sram_prepare_addr(++page);
  while( num_bytes >= 8 ) {
    sram_change_addr(page);
    sram_prepare_addr(++page);
    sram_write_byte(0, src[0]);
    sram_write_byte(1, src[1]);
    sram_write_byte(2, src[2]);
    sram_write_byte(3, src[3]);
    sram_write_byte(4, src[4]);
    sram_write_byte(5, src[5]);
    sram_write_byte(6, src[6]);
    sram_write_byte(7, src[7]);
    src += 8;
    num_bytes -= 8;
  }
  if( num_bytes ) {
    sram_change_addr(page);
    while( num_bytes ) {
      sram_write_byte(offset, src[0]);
      ++offset;
      ++src;
      --num_bytes;
    }
  }
}
